// =============================================================
// SPI master with 16-bit Baud Rate Register
//
// Author   : Klaus Kohl-Schoepe
// Date     : 30.06.2020
// File Name: mySPImaster.v
// Copyright (C) 2020 Klaus Kohl-Schoepe (kks@designin.de)
// =============================================================

// =============================================================
// Verilog SPI master with 16-bit Baud Rate Register (CLK / Baudrate / 2 - starting 1200 Baud @ 100MHz)
// It will use 8 bit with MSB first
//
// Short description:
//  spi_baudrate is the speed register
//  mosi_data and miso_data are the 8-bit data interface
//  miso_read and mosi_write are the read/write signal - should be set for read/write
//  spi_busy signalize that data are handled
// =============================================================

//`timescale 100 ns / 10 ns
`timescale 1 ns / 1 ps

// =============================================================
// Modul myUART
// =============================================================

module mySPImaster
(
  input             reset,        // Synchronous reset
  input             clk,          // The master clock
  input      [15:0] spi_baudrate, // Baudrate divider (speed = clk/baudrate/2)
  output reg        spi_clk,      // MOSI line
  output            spi_busy,     // Busy
  // MOSI
  output reg        spi_mosi,     // MOSI line
  input             spi_wr,       // write to SPI will start transfer
  input      [ 7:0] spi_out_data, // Byte to transmit
  // MISO
  input             spi_miso,     // MISO line
  output reg [ 7:0] spi_in_data   // Received data
);

  // Working register like baud rate and status
  localparam [1:0]
    X_IDLE = 2'd0,                // no operation / wait for start bit
    X_IN   = 2'd1,                // wait for next input (CLK to hi)
    X_OUT  = 2'd2;                // wait for next output (CLK to lo)

  reg [ 2:0] state = X_IDLE;      // state
  reg [15:0] baudcnt;             // Baud rate down counter
  reg [ 2:0] bitcnt;              // Bits 0..7  
  reg [ 7:0] datawi;              // Data (working register MISO)
  reg [ 7:0] datab;               // Data buffer for next output
  reg [ 7:0] datawo;              // Data (working register MOSI)
  reg        wr_f;                // New data available
  reg        wr_ff;               // wr have to go hi before next write

  // Assignment
  assign spi_busy = (state != X_IDLE);

  // Initialization
  initial begin
    state    = X_IDLE;          // Idle state
    wr_f     = 1'b0;            // No Data
    wr_ff    = 1'b0;            // Reset latch
    spi_clk  = 1'b0;            // Set clock to 0
    spi_mosi = 1'b0;            // Set data to 0
  end

// =============================================================
// Main Loop
// =============================================================
  // Check reset
  always @(posedge clk) begin
    // Reset restarts all
    if (reset) begin
      state    = X_IDLE;          // Idle state
      wr_f     = 1'b0;            // No Data
      wr_ff    = 1'b0;            // Reset latch
      spi_clk  = 1'b0;            // Set clock to 0
      spi_mosi = 1'b0;            // Set data to 0
    end

    // Check if data available for TX
    if (spi_wr) begin             // If request for transmit
      if (!wr_ff) begin           // still same write ?
        datab = spi_out_data;     // get data to buffer
        wr_f  = 1'b1;             // data available
        wr_ff = 1'b1;             // set lock
      end
    end else begin                // If no write
      wr_ff = 1'b0;               // reset acknowledge
    end

    // Decrement baud counter
    if(baudcnt) begin
        baudcnt = baudcnt - 1'd1;
    end

    // Begin State machine
    case (state)
      X_IDLE: begin
        if (wr_f) begin           // If data available
          spi_mosi = datab[7];    // Set output to MSB (clock is still 0)
          datawo   = {datab[6:0], 1'b0}; // Rest of data to working register MOSI
          wr_f     = 1'b0;        // Buffer now free
          bitcnt   = 3'd7;        // repeat next steps 8 times
          baudcnt  = spi_baudrate; // Wait one half cycle
          state    = X_IN;        // Wait for the first data to read
        end
      end
    
      X_IN: begin                 // Start bit included above
        if (!baudcnt) begin
		    datawi  = {datawi[6:0], spi_miso}; // Shift working register MISO and Read bit to LSB
          spi_clk = 1'b1;         // Set clock to 1
          baudcnt = spi_baudrate; // Wait one half cycle
          state   = X_OUT;        // Wait for the next data to write
		  end
      end
          
      X_OUT: begin
        if (!baudcnt) begin
          if (bitcnt) begin       // If not all 8 bits:
            spi_mosi = datawo[7]; // Set output to LSB
            datawo   = {datawo[6:0], 1'b0}; // shift working register MOSI
            baudcnt  = spi_baudrate; // Wait one half cycle
            bitcnt   = bitcnt - 1'b1; // next bit
            state    = X_IN;      // Wait for the first data to read
          end else begin
			   spi_in_data = datawi; // data ready
            state    = X_IDLE;    // ready for the next byte		    
          end
          spi_clk  = 1'b0;        // Set clock to 0
		  end
      end
    endcase
	 // End State Machine
  end

endmodule
